#ifndef __UBASIC_H__
#define __UBASIC_H__

#include <string.h>
#include <ctype.h>
#include <map>
#include <string>
#include <iostream>
using namespace std;

#define DEBUG 0

#if DEBUG
#define DEBUG_PRINTF(...)  printf(__VA_ARGS__)
#else
#define DEBUG_PRINTF(...)
#endif


class uBasic
{
public:
	uBasic() { init("10 print\nHello World"); }
	void init(const char *program)
	{
		parse_error = false;
		program_ptr = program;
		for_stack_ptr = gosub_stack_ptr = 0;
		tokenizer.init(program);
		ended = 0;
	}
	bool run(void)
	{
		if(!tokenizer.finished() && !ended && !parse_error)  
		{
			line_statement();
			return true;
		}
		else
		{
			DEBUG_PRINTF("uBASIC program finished\n");
			return false;
		}
	}

	#define MAX_VARNUM 26
	void set_variable(int varnum, int value)
	{
		if(varnum > 0 && varnum <= MAX_VARNUM) {
			variables[varnum] = value;
		}
	}
	int get_variable(int varnum)
	{
		if(varnum > 0 && varnum <= MAX_VARNUM) {
			return variables[varnum];
		}
		return 0;
	}
protected:
	enum TOKENS
	{
		TK_ERROR,
		TK_ENDOFINPUT,
		TK_NUMBER,
		TK_STRING,
		TK_VARIABLE,
		TK_LET,
		TK_PRINT,
		TK_IF,
		TK_THEN,
		TK_ELSE,
		TK_FOR,
		TK_TO,
		TK_NEXT,
		TK_GOTO,
		TK_GOSUB,
		TK_RETURN,
		TK_CALL,
		TK_END,
		TK_COMMA,
		TK_SEMICOLON,
		TK_PLUS,
		TK_MINUS,
		TK_AND,
		TK_OR,
		TK_ASTR,
		TK_SLASH,
		TK_MOD,
		TK_LEFTPAREN,
		TK_RIGHTPAREN,
		TK_LT,
		TK_GT,
		TK_EQ,
		TK_CR,
	};


	class Tokenizer
	{
	protected:
		int cur_line_number;
		int current_token;
		map <string, int> keywords;
		string keyword; // current found keyword 
	public:

		Tokenizer()
		{
			current_token = TK_ERROR;
			keywords["let"]=TK_LET;
			keywords["print"]=TK_PRINT;
			keywords["if"]=TK_IF;
			keywords["then"]=TK_THEN;
			keywords["else"]=TK_ELSE;
			keywords["for"]=TK_FOR;
			keywords["to"]=TK_TO;
			keywords["next"]=TK_NEXT;
			keywords["goto"]=TK_GOTO;
			keywords["gosub"]=TK_GOSUB;
			keywords["return"]=TK_RETURN;
			keywords["call"]=TK_CALL;
			keywords["end"]=TK_END;
		}
		char const *ptr, *nextptr;

#define MAX_NUMLEN 5


		int line_number() { return cur_line_number; }

		int singlechar(void)
		{
			if(*ptr == '\n') {
				return TK_CR;
			} else if(*ptr == ',') {
				return TK_COMMA;
			} else if(*ptr == ';') {
				return TK_SEMICOLON;
			} else if(*ptr == '+') {
				return TK_PLUS;
			} else if(*ptr == '-') {
				return TK_MINUS;
			} else if(*ptr == '&') {
				return TK_AND;
			} else if(*ptr == '|') {
				return TK_OR;
			} else if(*ptr == '*') {
				return TK_ASTR;
			} else if(*ptr == '/') {
				return TK_SLASH;
			} else if(*ptr == '%') {
				return TK_MOD;
			} else if(*ptr == '(') {
				return TK_LEFTPAREN;
			} else if(*ptr == ')') {
				return TK_RIGHTPAREN;
			} else if(*ptr == '<') {
				return TK_LT;
			} else if(*ptr == '>') {
				return TK_GT;
			} else if(*ptr == '=') {
				return TK_EQ;
			}
			return 0;
		}

		int get_next_token(void)
		{
			int i;

			DEBUG_PRINTF("get_next_token(): '%s'\n", ptr);

			if(*ptr == 0) 
			{
				return TK_ENDOFINPUT;
			}

			if(isdigit(*ptr)) 
			{
				for(i = 0; i < MAX_NUMLEN; ++i) 
				{
					if(!isdigit(ptr[i])) 
					{
						if(i > 0) 
						{
							nextptr = ptr + i;
							return TK_NUMBER;
						} 
						else 
						{
							DEBUG_PRINTF("get_next_token: error due to too short number\n");
							return TK_ERROR;
						}
					}

					if(!isdigit(ptr[i])) 
					{
						DEBUG_PRINTF("get_next_token: error due to malformed number\n");
						return TK_ERROR;
					}
				}
				DEBUG_PRINTF("get_next_token: error due to too long number\n");
				return TK_ERROR;
			} 
			else if(singlechar()) 
			{
				nextptr = ptr + 1;
				return singlechar();
			} 
			else if(*ptr == '"') 
			{
				nextptr = ptr;
				do 
				{
					++nextptr;
				} 
				while(*nextptr != '"');
				++nextptr;
				return TK_STRING;
			} 
			else 
			{
				nextptr = ptr;
				keyword="";
				do 
				{
					keyword += *nextptr;
					++nextptr;
				} 
				while(*nextptr != ' ' && *nextptr != '\n' && *nextptr != '\0');

				map<string,int>::iterator i = keywords.find(keyword);
				if(i!=keywords.end()) // found the token
					return i->second;
			}

			if(*ptr >= 'a' && *ptr <= 'z') 
			{
				nextptr = ptr + 1;
				return TK_VARIABLE;
			}


			return TK_ERROR;
		}

		void init(const char *program)
		{
			cur_line_number = 0;
			ptr = program;
			current_token = get_next_token();
		}

		int token(void)
		{
			return current_token;
		}

		void next(void)
		{

			if(finished()) 
			{
				return;
			}

			DEBUG_PRINTF("next: %p\n", nextptr);
			ptr = nextptr;
			// skip whitespaces
			while(*ptr == ' ' || *ptr == '\t') 
			{
				++ptr;
			}
			current_token = get_next_token();
			if(current_token==TK_CR)
				cur_line_number++;
			DEBUG_PRINTF("next: '%s' %d\n", ptr, current_token);
			return;
		}

		int num(void)
		{
			return atoi(ptr);
		}

		void tk_string(char *dest, int len)
		{
			char *string_end;
			int string_len;

			if(token() != TK_STRING) 
			{
				return;
			}
			string_end = (char*)strchr((ptr + 1), '"');
			if(string_end == NULL) 
			{
				return;
			}
			string_len = string_end - ptr - 1;
			if(len < string_len) {
				string_len = len;
			}
			memcpy(dest, ptr + 1, string_len);
			dest[string_len] = 0;
		}

		void error_print(void)
		{
			DEBUG_PRINTF("error_print: '%s'\n", ptr);
		}
		int finished(void)
		{
			return *ptr == 0 || current_token == TK_ENDOFINPUT;
		}
		int variable_num(void)
		{
			return *ptr - 'a';
		}
	};





	Tokenizer tokenizer;

	bool parse_error;

	char const *program_ptr;
#define MAX_STRINGLEN 40
	char strng[MAX_STRINGLEN];

#define MAX_GOSUB_STACK_DEPTH 10
	int gosub_stack[MAX_GOSUB_STACK_DEPTH];
	int gosub_stack_ptr;

	struct for_state 
	{
		int line_after_for;
		int for_variable;
		int to;
	};
#define MAX_FOR_STACK_DEPTH 4
	struct for_state for_stack[MAX_FOR_STACK_DEPTH];
	int for_stack_ptr;

	char variables[MAX_VARNUM];

	int ended;


	void accept(int token)
	{
		if(token != tokenizer.token())
		{
			DEBUG_PRINTF("unexpected Token. (expected %d, got %d)\n", token, tokenizer.token());
			tokenizer.error_print();
			//parse_error=true;
		}
		DEBUG_PRINTF("Expected %d, got it\n", token);
		tokenizer.next();
	}

	int varfactor(void)
	{
		int r;
		DEBUG_PRINTF("varfactor: obtaining %d from variable %d\n", variables[tokenizer.variable_num()], tokenizer.variable_num());
		r = get_variable(tokenizer.variable_num());
		accept(TK_VARIABLE);
		return r;
	}

	int factor(void)
	{
		int r;

		DEBUG_PRINTF("factor: token %d\n", tokenizer.token());
		switch(tokenizer.token())
		{
		case TK_NUMBER:
			r = tokenizer.num();
			DEBUG_PRINTF("factor: number %d\n", r);
			accept(TK_NUMBER);
			break;
		case TK_LEFTPAREN:
			accept(TK_LEFTPAREN);
			r = expr();
			accept(TK_RIGHTPAREN);
			break;
		default:
			r = varfactor();
			break;
		}
		return r;
	}

	int term(void)
	{
		int f1, f2;
		int op;

		f1 = factor();
		op = tokenizer.token();
		DEBUG_PRINTF("term: token %d\n", op);
		while(	op == TK_ASTR ||
			op == TK_SLASH ||
			op == TK_MOD)
		{
			tokenizer.next();
			f2 = factor();
			DEBUG_PRINTF("term: %d %d %d\n", f1, op, f2);
			switch(op) 
			{
			case TK_ASTR:
				f1 = f1 * f2;
				break;
			case TK_SLASH:
				f1 = f1 / f2;
				break;
			case TK_MOD:
				f1 = f1 % f2;
				break;
			}
			op = tokenizer.token();
		}
		DEBUG_PRINTF("term: %d\n", f1);
		return f1;
	}

	int expr(void)
	{
		int t1, t2;
		int op;

		t1 = term();
		op = tokenizer.token();
		DEBUG_PRINTF("expr: token %d\n", op);
		while(op == TK_PLUS ||
			op == TK_MINUS ||
			op == TK_AND ||
			op == TK_OR) 
		{
			tokenizer.next();
			t2 = term();
			DEBUG_PRINTF("expr: %d %d %d\n", t1, op, t2);
			switch(op) 
			{
			case TK_PLUS:
				t1 = t1 + t2;
				break;
			case TK_MINUS:
				t1 = t1 - t2;
				break;
			case TK_AND:
				t1 = t1 & t2;
				break;
			case TK_OR:
				t1 = t1 | t2;
				break;
			}
			op = tokenizer.token();
		}
		DEBUG_PRINTF("expr: %d\n", t1);
		return t1;
	}

	int relation(void)
	{
		int r1, r2;
		int op;

		r1 = expr();
		op = tokenizer.token();
		DEBUG_PRINTF("relation: token %d\n", op);
		while(op == TK_LT ||
			op == TK_GT ||
			op == TK_EQ) {
				tokenizer.next();
				r2 = expr();
				DEBUG_PRINTF("relation: %d %d %d\n", r1, op, r2);
				switch(op) {
	case TK_LT:
		r1 = r1 < r2;
		break;
	case TK_GT:
		r1 = r1 > r2;
		break;
	case TK_EQ:
		r1 = r1 == r2;
		break;
				}
				op = tokenizer.token();
		}
		return r1;
	}

	void jump_line_num(int linenum)
	{
		tokenizer.init(program_ptr);
		while(tokenizer.token() != TK_ENDOFINPUT)
		{
			if(tokenizer.line_number() == linenum) 
			{
				DEBUG_PRINTF("jump_line_num: Found line %d\n", tokenizer.line_num());
				return;
			}
			tokenizer.next();
		}
	}
	void jump_line_label(int linenum)
	{
		tokenizer.init(program_ptr);
		while(tokenizer.num() != linenum) {
			do {
				do {
					tokenizer.next();
				} while(tokenizer.token() != TK_CR &&
					tokenizer.token() != TK_ENDOFINPUT);
				if(tokenizer.token() == TK_CR) {
					tokenizer.next();
				}
			} while(tokenizer.token() != TK_NUMBER);
			DEBUG_PRINTF("jump_linenum: Found line %d\n", tokenizer.num());
		}
	}

	void goto_statement(void)
	{
		accept(TK_GOTO);
		jump_line_label(tokenizer.num());
	}

	void print_statement(void)
	{
		accept(TK_PRINT);
		do {
			DEBUG_PRINTF("Print loop\n");
			if(tokenizer.token() == TK_STRING) {
				tokenizer.tk_string(strng, sizeof(strng));
				printf("%s", strng);
				tokenizer.next();
			} else if(tokenizer.token() == TK_COMMA) {
				printf(" ");
				tokenizer.next();
			} else if(tokenizer.token() == TK_SEMICOLON) {
				tokenizer.next();
			} else if(tokenizer.token() == TK_VARIABLE ||
				tokenizer.token() == TK_NUMBER) {
					printf("%d", expr());
			} else {
				break;
			}
		} while(tokenizer.token() != TK_CR &&
			tokenizer.token() != TK_ENDOFINPUT);
		printf("\n");
		DEBUG_PRINTF("End of print\n");
		tokenizer.next();
	}

	void if_statement(void)
	{
		int r;

		accept(TK_IF);

		r = relation();
		DEBUG_PRINTF("if_statement: relation %d\n", r);
		accept(TK_THEN);
		if(r) {
			statement();
		} else {
			do {
				tokenizer.next();
			} while(tokenizer.token() != TK_ELSE &&
				tokenizer.token() != TK_CR &&
				tokenizer.token() != TK_ENDOFINPUT);
			if(tokenizer.token() == TK_ELSE) {
				tokenizer.next();
				statement();
			} else if(tokenizer.token() == TK_CR) {
				tokenizer.next();
			}
		}
	}

	void let_statement(void)
	{
		int var;

		var = tokenizer.variable_num();

		accept(TK_VARIABLE);
		accept(TK_EQ);
		set_variable(var, expr());
		DEBUG_PRINTF("let_statement: assign %d to %d\n", variables[var], var);
		accept(TK_CR);

	}

	void gosub_statement(void)
	{
		int linenum;
		accept(TK_GOSUB);
		linenum = tokenizer.num();
		accept(TK_NUMBER);
		accept(TK_CR);
		if(gosub_stack_ptr < MAX_GOSUB_STACK_DEPTH) {
			gosub_stack[gosub_stack_ptr] = tokenizer.line_number();
			gosub_stack_ptr++;
			jump_line_label(linenum);
		} else {
			DEBUG_PRINTF("gosub_statement: gosub stack exhausted\n");
		}
	}

	void return_statement(void)
	{
		accept(TK_RETURN);
		if(gosub_stack_ptr > 0) 
		{
			gosub_stack_ptr--;
			jump_line_num(gosub_stack[gosub_stack_ptr]);
		} 
		else 
		{
			DEBUG_PRINTF("return_statement: non-matching return\n");
		}
	}

	void next_statement(void)
	{
		int var;

		accept(TK_NEXT);
		var = tokenizer.variable_num();
		accept(TK_VARIABLE);
		if(for_stack_ptr > 0 &&
			var == for_stack[for_stack_ptr - 1].for_variable) {
				set_variable(var,
					get_variable(var) + 1);
				if(get_variable(var) <= for_stack[for_stack_ptr - 1].to) {
					jump_line_num(for_stack[for_stack_ptr - 1].line_after_for);
				} else {
					for_stack_ptr--;
					accept(TK_CR);
				}
		} else {
			DEBUG_PRINTF("next_statement: non-matching next (expected %d, found %d)\n", for_stack[for_stack_ptr - 1].for_variable, var);
			accept(TK_CR);
		}

	}
	void for_statement(void)
	{
		int for_variable, to;

		accept(TK_FOR);
		for_variable = tokenizer.variable_num();
		accept(TK_VARIABLE);
		accept(TK_EQ);
		set_variable(for_variable, expr());
		accept(TK_TO);
		to = expr();
		accept(TK_CR);

		if(for_stack_ptr < MAX_FOR_STACK_DEPTH) 
		{
			for_stack[for_stack_ptr].line_after_for = tokenizer.line_number();
			for_stack[for_stack_ptr].for_variable = for_variable;
			for_stack[for_stack_ptr].to = to;
			DEBUG_PRINTF("for_statement: new for, var %d to %d\n", for_stack[for_stack_ptr].for_variable, for_stack[for_stack_ptr].to);
			for_stack_ptr++;
		} 
		else 
		{
			DEBUG_PRINTF("for_statement: for stack depth exceeded\n");
		}
	}
	void end_statement(void)
	{
		accept(TK_END);
		ended = 1;
	}
	void statement(void)
	{
		int token;

		token = tokenizer.token();

		switch(token) {
	case TK_PRINT:
		print_statement();
		break;
	case TK_IF:
		if_statement();
		break;
	case TK_GOTO:
		goto_statement();
		break;
	case TK_GOSUB:
		gosub_statement();
		break;
	case TK_RETURN:
		return_statement();
		break;
	case TK_FOR:
		for_statement();
		break;
	case TK_NEXT:
		next_statement();
		break;
	case TK_END:
		end_statement();
		break;
	case TK_LET:
		accept(TK_LET);
		/* Fall through. */
	case TK_VARIABLE:
		let_statement();
		break;
	default:
		DEBUG_PRINTF("ubasic.c: statement(): not implemented %d\n", token);
		tokenizer.next();
		//parse_error=true;
		}
	}
	void line_statement(void)
	{
		DEBUG_PRINTF("----------- Line number %d ---------\n", tokenizer.num());
		/*    current_linenum = tokenizer.num();*/
		//accept(TK_NUMBER);
		statement();
		return;
	}
};

#endif /* __UBASIC_H__ */
